/**
 * \file pappsomspp/processing/specpeptidoms/scenario.cpp
 * \date 24/03/2025
 * \author Aurélien Berthier
 * \brief backtracking of 2nd alignment
 *
 * C++ implementation of the SpecPeptidOMS algorithm described in :
 * (1) Benoist, É.; Jean, G.; Rogniaux, H.; Fertin, G.; Tessier, D. SpecPeptidOMS Directly and
 * Rapidly Aligns Mass Spectra on Whole Proteomes and Identifies Peptides That Are Not Necessarily
 * Tryptic: Implications for Peptidomics. J. Proteome Res. 2025.
 * https://doi.org/10.1021/acs.jproteome.4c00870.
 */

/*
 * Copyright (c) 2025 Aurélien Berthier
 * <aurelien.berthier@ls2n.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "scenario.h"
#include <QDebug>
#include <pappsomspp/core/exception/exceptionoutofrange.h>

// TODO : reserve space for origin matrix
// Scenario::Scenario()
// {
// }

void
pappso::specpeptidoms::Scenario::saveOrigin(std::size_t current_row,
                                            std::size_t current_column,
                                            std::size_t previous_row,
                                            std::size_t previous_column,
                                            int score,
                                            AlignType alignment_type)
{
  if(score > m_best_alignment.second)
    {
      m_best_alignment = {{current_row, current_column, alignment_type}, score};
    }
  try
    {
      m_origin_matrix.at(current_row).at(current_column) = {
        previous_row, previous_column, alignment_type};
    }
  catch(const std::exception &error)
    {
      int max_column_size = 0;
      if(current_row < m_origin_matrix.size())
        {
          max_column_size = m_origin_matrix.at(current_row).size();
        }
      throw pappso::ExceptionOutOfRange(
        QObject::tr(
          "Scenario::saveOrigin failed for current_row %1 (size %2)  current_column %3 (size %4)")
          .arg(current_row)
          .arg(m_origin_matrix.size())
          .arg(current_column)
          .arg(max_column_size));
    }
}

pappso::specpeptidoms::Scenario::Scenario()
{
}

pappso::specpeptidoms::Scenario::~Scenario()
{
}

void
pappso::specpeptidoms::Scenario::reserve(std::size_t n_rows, std::size_t n_columns)
{
  m_origin_matrix = std::vector<std::vector<ScenarioCell>>(
    n_rows, std::vector<ScenarioCell>(n_columns, {0, 0, AlignType::init}));
}

std::pair<std::vector<pappso::specpeptidoms::ScenarioCell>, int>
pappso::specpeptidoms::Scenario::getBestAlignment() const
{
  std::pair<std::vector<ScenarioCell>, int> best_alignment;
  ScenarioCell current_cell   = m_best_alignment.first;
  current_cell.alignment_type = AlignType::init;
  while(current_cell.previous_column > 0 && current_cell.previous_row > 0)
    {
      best_alignment.first.push_back(current_cell);
      current_cell = m_origin_matrix.at(current_cell.previous_row).at(current_cell.previous_column);
    }
  best_alignment.first.push_back(current_cell);
  best_alignment.second = m_best_alignment.second;
  return best_alignment;
}

int
pappso::specpeptidoms::Scenario::getBestScore() const
{
  return m_best_alignment.second;
}

void
pappso::specpeptidoms::Scenario::resetScenario()
{
  // m_origin_matrix.clear();
  m_best_alignment = {{0, 0, AlignType::init}, -15};
}

void
pappso::specpeptidoms::Scenario::clear()
{
  m_origin_matrix.clear();
  m_best_alignment = {{0, 0, AlignType::init}, -15};
}
